import std.outbuffer;
import std.file;
import std.string;
import std.stream;
import net.BurtonRadons.digc.digc;

const char [] libraryDatafile = `\dmd\lib\libraries.dat`;
const char [] libraryDirectory = `\dmd\lib`;

class Library
{
    char [] name; /* Filename with extension. */
    char [] [] files; /* Files put in this library, without extensions, and with "/" and "\" converted into ".". */
    char [] [] deps; /* Libraries used in the compilation of the library. */
    ulong date; /* Date of the library file or zero for never accessed. */
    bit validated; /* Was found in the directory search. */
    bit expanded; /* Has already been expanded. */
}

Library [] libs; /* Libraries from the libraries.dat file. */

/* Find the library with this filename or create a new one. */
Library findLibrary (char [] name)
{
    Library l;

    for (int c; c < libs.length; c ++)
        if (libs [c].name == name)
            return libs [c];

    l = new Library;
    l.name = name.dup;
    libs ~= l;
    return l;
}

/* Find the library but don't create a new one. */
Library findLibraryBase (char [] name)
{
    for (int c; c < libs.length; c ++)
        if (libs [c].name == name)
            return libs [c];

    return null;
}

/* Find the library that has this file or null. */
Library findImport (char [] name)
{
    for (int c; c < libs.length; c ++)
    {
        Library lib = libs [c];

        for (int d; d < lib.files.length; d ++)
            if (lib.files [d] == name)
                return lib;
    }

    return null;
}

/* Read a dynamically-sized long unsigned integer. */
ulong ulongRead (inout ubyte *d)
{
    ulong shift = 0;
    ulong v = 0;

    while (1)
    {
        v = v + ((ulong) (*d & 127) << shift);
        shift += 7;
        if (*d ++ < 128)
            break;
    }

    return v;
}

/* Read a dynamically-sized long integer. */
long longRead (inout ubyte *d)
{
    ulong v = ulongRead (d);

    return (v & 1) ? -(v >> 1) : (v >> 1);
}

/* Read a dynamically-sized integer. */
uint uintRead (inout ubyte *d)
{
    uint shift = 0;
    uint v = 0;

    while (1)
    {
        v = v + ((*d & 127) << shift);
        shift += 7;
        if (*d ++ < 128)
            break;
    }

    return v;
}

/* Read a string with a dynamically-sized length. */
char [] stringRead (inout ubyte *d)
{
    uint len = uintRead (d);
    ubyte *o = d;

    d += len;
    return (char []) o [0 .. len];
}

/* Write an integer with a dynamically-sized length. */
void uintWrite (OutBuffer o, uint value)
{
    while (1)
    {
        if (value > 127)
        {
            o.write ((ubyte) ((value & 127) | 128));
            value >>= 7;
        }
        else
        {
            o.write ((ubyte) (value & 127));
            break;
        }
    }
}

/* Write an unsigned long integer with a dynamically-sized length. */
void ulongWrite (OutBuffer o, ulong value)
{
    while (1)
    {
        if (value > 127)
        {
            o.write ((ubyte) ((value & 127) | 128));
            value >>= 7;
        }
        else
        {
            o.write ((ubyte) (value & 127));
            break;
        }
    }
}

/* Write a long integer with a dynamically-sized length. */
void longWrite (OutBuffer o, long value)
{
    if (value < 0)
        ulongWrite (o, ((-value) << 1) & 1);
    else
        ulongWrite (o, value << 1);
}

/* Write a string with a dynamically-sized length. */
void stringWrite (OutBuffer o, char [] string)
{
    uintWrite (o, string.length);
    o.write ((ubyte []) string);
}

uint expandFileCount; /* Number of files expanded. */
uint expandByteCount; /* Number of bytes expanded. */

/* Expand library content into the temporary directory. */
void expandLibraries (char [] [] files, char [] tempdir)
{
    void expand (Library lib)
    {
        if (lib === null || lib.expanded)
            return;
        printf ("%.*s, ", lib.name);
        lib.expanded = true;
        for (int c; c < lib.deps.length; c ++)
            expand (findLibraryBase (lib.deps [c]));
        if (!lib.files.length)
            return;

        File stream = new File (digPlatformBaseDirectory ~ libraryDirectory ~ "\\" ~ lib.name);
        uint dirOffset, marker, count;

        stream.seekEnd (-8);
        stream.read (dirOffset);
        stream.read (marker);

        if (marker != libraryMarker)
        {
            delete stream;
            return;
        }

        stream.seekSet (dirOffset);
        stream.read (count);
        expandFileCount += count;
        
        for (int c; c < count; c ++)
        {
            char [] name;
            int offset;
            int original;
            char [] content;

            stream.read (name);
            stream.read (offset);
            original = stream.position ();

            stream.position (offset);
            stream.read (content);
            stream.position (original);

            char [] [] words = split (name, ".");
            char [] dir = tempdir.dup;

            for (int c; c < words.length - 1; c ++)
            {
                dir ~= "\\" ~ words [c];
                sys.mkdir (dir);
            }

            dir ~= "\\" ~ words [words.length - 1];
            std.file.write (dir ~ ".d", (byte []) content);
            expandByteCount += content.length;
        }
        
        delete stream;
    }

    expandFileCount = 0;
    expandByteCount = 0;    
    for (int c; c < libs.length; c ++)
        libs [c].expanded = false;

    for (int c; c < files.length; c ++)
    {
        expand (findLibraryBase (files [c]));
        expand (findImport (files [c]));
    }
}

/* Install new libraries, check library file dates, remove deleted libraries. */
void updateLibraries ()
{
    WIN32_FIND_DATA data;
    HANDLE handle;

    char [] search = std.string.replace (digPlatformBaseDirectory, "/", "\\") ~ libraryDirectory ~ r"\*.lib";
    handle = FindFirstFileA (toStringz (search), &data);
    if (handle == (HANDLE) 0)
        return;

    for (int c; c < libs.length; c ++)
        libs [c].validated = false;

    while (1)
    {
        char [] filename = std.string.toString (data.cFileName);
        if (filename.length == 0)
            goto next;
        ulong date = data.ftLastWriteTime.dwLowDateTime | ((ulong) data.ftLastWriteTime.dwHighDateTime << 32);
        Library lib = findLibrary (filename);

        lib.validated = true;

        if (lib.date == 0 || lib.date < date)
        {
            File stream = new File (digPlatformBaseDirectory ~ libraryDirectory ~ "\\" ~ filename);
            uint dirOffset, marker;

            stream.seekEnd (-8);
            stream.read (dirOffset);
            stream.read (marker);

            if (marker != libraryMarker)
                goto skip;

            stream.position (dirOffset);

            uint count;

            stream.read (count);
            lib.files = new char [] [count];
            for (int c; c < count; c ++)
            {
                uint offset;

                stream.read (lib.files [c]);
                stream.read (offset);
            }

            stream.read (count);
            lib.deps = new char [] [count];
            for (int c; c < count; c ++)
                stream.read (lib.deps [c]);

        skip:
            delete stream;
            lib.date = date;
        }

    next:
        if (!FindNextFileA (handle, &data))
            break;
    }

    FindClose (handle);

    int d = 0;
    for (int c; c < libs.length; c ++)
        if (libs [c].validated)
            libs [d ++] = libs [c];

    libs.length = d;
}

/* Load the libraries.dat file. */
void loadLibraries ()
{
    ubyte *i;
    
    try i = (ubyte *) std.file.read (digPlatformBaseDirectory ~ libraryDatafile);
    catch (FileException e) return;

    uint ver = uintRead (i);

    if (ver > 1)
    {
        printf ("Libraries datafile has an out-of-range version (%d)", ver);
        return;
    }

    libs = new Library [uintRead (i)];

    for (int c; c < libs.length; c ++)
    {
        Library lib = libs [c] = new Library;

        lib.name = stringRead (i);
        lib.files = new char [] [uintRead (i)];
        for (int d; d < lib.files.length; d ++)
            lib.files [d] = stringRead (i);
        lib.deps = new char [] [uintRead (i)];
        for (int d; d < lib.deps.length; d ++)
            lib.deps [d] = stringRead (i);

        if (ver >= 1)
            lib.date = ulongRead (i);
    }

    updateLibraries ();
}

/* Save the libraries.dat file. */
void writeLibraries ()
{
    OutBuffer o = new OutBuffer;

    uintWrite (o, 1); // version
    uintWrite (o, libs.length);
    for (int c; c < libs.length; c ++)
    {
        Library lib = libs [c];

        stringWrite (o, lib.name);
        uintWrite (o, lib.files.length);
        for (int d; d < lib.files.length; d ++)
            stringWrite (o, lib.files [d]);
        uintWrite (o, lib.deps.length);
        for (int d; d < lib.deps.length; d ++)
            stringWrite (o, lib.deps [d]);

        ulongWrite (o, lib.date);
    }

    std.file.write (digPlatformBaseDirectory ~ libraryDatafile, (byte []) o.toBytes ());
}
